/** @file gtk_joysetbox.cpp

	SHARP X68000 Emulator 'eCZ-600'
	Skelton for retropc emulator

	@author Sasaji
	@date   2023.01.07 -

	@brief [ joypad setting box ]
*/

#include "gtk_joysetbox.h"
#include "../../emumsg.h"
#include "../../msgs.h"
#include "../../labels.h"
#include "../gui_keybinddata.h"
#include "../../utility.h"
#include "../../keycode.h"

extern EMU *emu;

namespace GUI_GTK_X11
{

JoySettingBox::JoySettingBox(GUI *new_gui)
		: KeybindControlBox(new_gui)
{
}

JoySettingBox::~JoySettingBox()
{
}

bool JoySettingBox::Show(GtkWidget *parent_window)
{
	KeybindControlBox::Show(parent_window);

	if (dialog) return true;

	create_dialog(window, CMsg::Joypad_Setting);
	add_accept_button(CMsg::OK);
	add_reject_button(CMsg::Cancel);

	GtkWidget *cont = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
	GtkWidget *vboxall;
//	GtkWidget *vboxbox;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *btn;
	GtkWidget *lbl;

	char label[128];

	GtkWidget *boxall = create_vbox(cont);
	GtkWidget *hboxall = create_hbox(boxall);

	int tx = 80;
	int sx = 160;
	int sy = 16;
	int cs = 0;

	//

#if defined(USE_PIAJOYSTICK) || defined(USE_KEY2JOYSTICK)
	for(int i=0; i<MAX_JOYSTICKS; i++) {
		vbox = create_vbox(hboxall, cs);

		UTILITY::sprintf(label, sizeof(label), CMSG(JoypadVDIGIT), i + 1);
		create_label(vbox, label);
		int val;
		hbox = create_hbox(vbox, cs);
#ifdef USE_JOYSTICK_TYPE
		val = pConfig->joy_type[i];
		com[i] = create_combo_box(hbox, "", LABELS::joypad_type, val);
#endif
		hbox = create_hbox(vbox, cs);
		create_label(hbox, CMsg::Button_Mashing_Speed);
		lbl = create_label(hbox, " ");
		gtk_widget_set_size_request(lbl, 32, sy);
		create_label(hbox, "0 <-> 3");

		GtkWidget *scroll = create_scroll_win(vbox, 240, 320);
		GtkWidget *svbox = create_vbox(scroll, cs);
		for(int k=0; k<KEYBIND_JOY_BUTTONS; k++) {
			int kk = k + VM_JOY_LABEL_BUTTON_A;
			if (kk >= VM_JOY_LABELS_MAX) {
				scale[i][k] = NULL;
				continue;
			}

			hbox = create_hbox(svbox, cs);
			CMsg::Id id = (CMsg::Id)cVmJoyLabels[kk].id;
			lbl = create_label(hbox, id);
			gtk_widget_set_size_request(lbl, tx, sy);
			val = pConfig->joy_mashing[i][k];
			int n = i * KEYBIND_JOY_BUTTONS + k;
			scale[i][k] = create_scale_box(hbox, 0, 3, val, false, n);
			gtk_scale_set_value_pos(GTK_SCALE(scale[i][k]), GTK_POS_RIGHT);
			gtk_widget_set_size_request(scale[i][k], sx, sy);
		}

		hbox = create_hbox(vbox, cs);
		create_label(hbox, CMsg::Analog_to_Digital_Sensitivity);
		lbl = create_label(hbox, " ");
		gtk_widget_set_size_request(lbl, 32, sy);
		create_label(hbox, "0 <-> 10");

		scroll = create_scroll_win(vbox, 240, 180);
		svbox = create_vbox(scroll, cs);
		for(int k=0; k<6; k++) {
			hbox = create_hbox(svbox, cs);
			CMsg::Id id = LABELS::joypad_axis[k];
			lbl = create_label(hbox, id);
			gtk_widget_set_size_request(lbl, tx, sy);
			val = 10 - pConfig->joy_axis_threshold[i][k];
			int n = i * 6 + k;
			axis[i][k] = create_scale_box(hbox, 0, 10, val, false, n);
			gtk_scale_set_value_pos(GTK_SCALE(axis[i][k]), GTK_POS_RIGHT);
			gtk_widget_set_size_request(axis[i][k], sx, sy);
		}
	}
#endif

	// create notebook tab
	vboxall = create_vbox(hboxall);
	notebook = create_notebook(vboxall);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(notebook), TRUE);

	int tab_offset = KeybindData::JS_TABS_MIN;
	for(int tab_num=tab_offset; tab_num < KeybindData::JS_TABS_MAX; tab_num++) {
		KeybindDataControl *kc = new KeybindDataControl();
		ctrls.Add(kc);

		// add note(tab) to notebook
		GtkWidget *vboxtab = create_vbox(NULL);
//		add_note(notebook, vboxtab, LABELS::joysetting_tab[tab_num-tab_offset]);
		UTILITY::sprintf(label, sizeof(label), "%d", tab_num + 1 - tab_offset);
		add_note(notebook, vboxtab, label);
		create_label(vboxtab, LABELS::joysetting_tab[tab_num-tab_offset]);

		kc->Create(this, vboxtab, tab_num, 280, 320
			, G_CALLBACK(OnKeyDown), G_CALLBACK(OnDoubleClick), G_CALLBACK(OnFocusIn));

#if 0
		if (!kc->AddCombiCheckButton(vboxtab, tab_num)) {
			create_label(vboxtab, "");
		}
#endif
#if defined(USE_PIAJOYSTICK) || defined(USE_KEY2PIAJOYSTICK)
		if (tab_num == Keybind::TAB_JOY2JOY) {
# ifdef USE_JOYSTICKBIT
			chkPiaJoyNeg = create_check_box(vboxtab, CMsg::Signals_are_negative_logic, FLG_PIAJOY_NEGATIVE != 0);
			hbox = create_hbox(vboxtab, cs);
			create_label(hbox, CMsg::Connect_to_);
			create_radio_box(hbox, LABELS::joysetting_opts, Config::PIAJOY_CONN_TO_MAX, radPiaJoyConn, pConfig->piajoy_conn_to);
# else
			chkPiaJoyNoIrq = create_check_box(vboxtab, CMsg::No_interrupt_caused_by_pressing_the_button, FLG_PIAJOY_NOIRQ != 0);
# endif
		}
#endif
#if defined(USE_PSGJOYSTICK) || defined(USE_KEY2PSGJOYSTICK)
		if (tab_num == Keybind::TAB_JOY2JOYB) {
# ifdef USE_JOYSTICKBIT
			chkPsgJoyNeg = create_check_box(vboxtab, CMsg::Signals_are_negative_logic, FLG_PSGJOY_NEGATIVE != 0);
# endif
		}
#endif
	}

	hbox = create_hbox(vboxall, cs);
	vbox = create_vbox(hbox, cs);
	btn = create_button(vbox, CMsg::Load_Default, G_CALLBACK(OnClickLoadDefaultJ));
	hbox = create_hbox(vboxall, cs);
	vbox = create_vbox(hbox, cs);
	for(int i=0; i<KEYBIND_PRESETS; i++) {
		UTILITY::sprintf(label, sizeof(label), CMSG(Load_Preset_VDIGIT), i+1);
		btn = create_button(vbox, label, G_CALLBACK(OnClickLoadPresetJ));
		g_object_set_data(G_OBJECT(btn), "num", (gpointer)(intptr_t)i);
	}
	vbox = create_vbox(hbox, cs);
	for(int i=0; i<KEYBIND_PRESETS; i++) {
		UTILITY::sprintf(label, sizeof(label), CMSG(Save_Preset_VDIGIT), i+1);
		btn = create_button(vbox, label, G_CALLBACK(OnClickSavePresetJ));
		g_object_set_data(G_OBJECT(btn), "num", (gpointer)(intptr_t)i);
	}

	ShowAfter(vboxall);

	emu->set_pause(1, true);

	return true;
}

void JoySettingBox::Hide()
{
	KeybindControlBox::Hide();
	notebook = NULL;
	emu->set_pause(1, false);
}

bool JoySettingBox::SetData()
{
	KeybindControlBox::SetData();

#if defined(USE_PIAJOYSTICK) || defined(USE_KEY2JOYSTICK)
	for(int i=0; i<MAX_JOYSTICKS; i++) {
#ifdef USE_JOYSTICK_TYPE
		pConfig->joy_type[i] = (int)get_combo_sel_num(com[i]);
#endif
		for(int k=0; k<KEYBIND_JOY_BUTTONS; k++) {
			GtkWidget *widget = scale[i][k];
			if (!widget) continue;
			pConfig->joy_mashing[i][k] = (int)get_scale_value(widget);
		}
		for(int k = 0; k < 6; k++) {
			GtkWidget *widget = axis[i][k];
			if (!widget) continue;
			pConfig->joy_axis_threshold[i][k] = 10 - (int)get_scale_value(widget);
		}
	}
	emu->modify_joy_mashing();
	emu->modify_joy_threshold();
#ifdef USE_JOYSTICK_TYPE
	// will change joypad type in emu thread
	emumsg.Send(EMUMSG_ID_MODIFY_JOYTYPE);
#endif
#if defined(USE_PIAJOYSTICK) || defined(USE_KEY2PIAJOYSTICK)
# ifdef USE_JOYSTICKBIT
	BIT_ONOFF(pConfig->misc_flags, MSK_PIAJOY_NEGATIVE, get_check_state(chkPiaJoyNeg));
	pConfig->piajoy_conn_to = get_radio_state_idx(radPiaJoyConn, Config::PIAJOY_CONN_TO_MAX);
# else
	BIT_ONOFF(pConfig->misc_flags, MSK_PIAJOY_NOIRQ, get_check_state(chkPiaJoyNoIrq));
# endif
#endif
#if defined(USE_PSGJOYSTICK) || defined(USE_KEY2PSGJOYSTICK)
# ifdef USE_JOYSTICKBIT
	BIT_ONOFF(pConfig->misc_flags, MSK_PSGJOY_NEGATIVE, get_check_state(chkPsgJoyNeg));
# endif
#endif
#endif
	return true;
}

#if 0
void JoySettingBox::OnChangeValue(GtkWidget *widget, gpointer user_data)
{
#if defined(USE_PIAJOYSTICK) || defined(USE_KEY2JOYSTICK)
	JoySettingBox *obj = (JoySettingBox *)user_data;
	int i = (int)(intptr_t)g_object_get_data(G_OBJECT(widget), "num");
	int k = i % KEYBIND_JOY_BUTTONS;
	i /= KEYBIND_JOY_BUTTONS;
	pConfig->joy_mashing[i][k] = (int)obj->get_scale_value(widget);
	emu->set_joy_mashing();
#endif
}
#endif

}; /* namespace GUI_GTK_X11 */


